#ifndef TMMLPTEALPAITAFNFAL_H
#define TMMLPTEALPAITAFNFAL_H

/*
STATEMENT               = [LINE_NUMBER] CONTROL_LOGIC | MATH_STATEMENT | IO_STATEMENTS | DECLARATION.
DECLARATION             = 'DECLARE' NUMERIC 'AS' NAME. 
NAME                    = NC { (NC | DECINT) }.
NC                      = <non-numeric printable ASCII char>
CONTROL_LOGIC           = BRANCH_STATEMENT | IF_FAMILY | WHILE_FAMILY.
IF_FAMILY               = IF_THEN | IF_THEN_ELSE | IF_THEN_UNLESS | IF_THEN_PROVIDED.
WHILE_FAMILY            = WHILE_DO | WHILE_DO_UNLESS | WHILE_DO_PROVIDED | UNLESS_DO | REPEAT_UNTIL | REPEAT_UNLESS | DO_WHILE | 
                          DO_UNTIL | DO_UNLESS | UNTIL_DO.
IF_THEN                 = 'IF' BOOLEAN_EXPRESSION 'THEN' STATEMENT.
IF_THEN_ELSE            = 'IF' BOOLEAN_EXPRESSION 'THEN' STATEMENT 'ELSE' STATEMENT.
IF_THEN_UNLESS          = 'IF' BOOLEAN_EXPRESSION 'THEN' STATEMENT 'UNLESS' BOOLEAN_EXPRESSION.
IF_THEN_PROVIDED        = 'IF' BOOLEAN_EXPRESSION 'THEN' STATEMENT 'PROVIDED' 'THAT' BOOLEAN_EXPRESSION.
WHILE_DO                = 'WHILE' BOOLEAN_EXPRESSION 'DO' STATEMENT.
WHILE_DO_UNLESS         = 'WHILE' BOOLEAN_EXPRESSION 'DO' STATEMENT 'UNLESS' BOOLEAN_EXPRESSION.
WHILE_DO_PROVIDED       = 'WHILE' BOOLEAN_EXPRESSION 'DO' STATEMENT 'PROVIDED' BOOLEAN_EXPRESSION.
UNLESS_DO               = 'UNLESS' BOOLEAN_EXPRESSION 'DO' STATEMENT.
UNTIL_DO                = 'UNTIL' BOOLEAN_EXPRESSION 'DO' STATEMENT.
REPEAT_UNTIL            = 'REPEAT' STATEMENT 'UNTIL' BOOLEAN_EXPRESSION.
REPEAT_UNLESS           = 'REPEAT' STATEMENT 'UNLESS' BOOLEAN_EXPRESSION.
DO_WHILE                = 'DO' STATEMENT 'WHILE' BOOLEAN_EXPRESSION.
DO_UNTIL                = 'DO' STATEMENT 'UNTIL' BOOLEAN_EXPRESSION.
DO_UNLESS               = 'DO' STATEMENT 'UNLESS' BOOLEAN_EXPRESSION.
IO_STATEMENTS           = READ | WRITE. "
WRITE                   = 'WRITE' ('CHAR'|'INTEGER') NUMERIC.
READ                    = 'READ' ('CHAR'|'INTEGER') NUMERIC.
BOOLEAN_EXPRESSION      = NUMERIC [ ('>'|'<'|'='|'<>'|'<='|'>=') NUMERIC].
MATH_STATEMENT          = ADD | SUB | MUL | MOD | DIV | COPY | NAND.
ADD                     = 'ADD' NUMERIC 'TO' NUMERIC. 
COPY                    = 'COPY' NUMERIC 'TO' NUMERIC. 
SUB                     = 'SUB' NUMERIC 'FROM' NUMERIC. 
MUL                     = 'MUL' NUMERIC 'WITH' NUMERIC. 
NAND                    = 'NAND' NUMERIC 'WITH' NUMERIC. 
MOD                     = 'MOD' NUMERIC 'BY' NUMERIC. 
DIV                     = 'DIV' NUMERIC 'BY' NUMERIC. 
NUMERIC                 = NAME | INTEGER | 'CELL' INTEGER ['INDIRECT'].
BRANCH_STATEMENT        = GOTO | GOSUB | STOP | RETURN.
STOP                    = 'STOP'.
RETURN                  = 'RETURN'.
GOSUB                   = 'GOSUB' NUMERIC.
GOTO                    = 'GOTO' NUMERIC.
LINE_NUMBER             = 'LINE' INTEGER ':'.
INTEGER                 = DECINT { DECINT }.
DECINT                  = '0' .. '9'.
*/
class RNG 
    {
    public:
        RNG();
        long GetRandomNumber();

    protected:    
        long k, idum, idum2, iy, j;
        long iv[64];
    };

#define isValidChar(c)  ((c>='0')&&(c<='z')||(c=='_')||(c=='$')||(c=='@')||(c=='')||(c=='?')||(c=='.')||(c==',')||(c=='!'))
#define isWhitespace(c)  ((c==' ')||(c=='\t'))
#define isKeywordSeparator(c) (isWhitespace(c) || !isValidChar(c))
#define SkipWhitespaces() while(isWhitespace(*m_pszCursor)) m_pszCursor++

class INSTRUCTION;

#define ACCESS_IMMEDIATE            0
#define ACCESS_MEMORY               1
#define ACCESS_MEMORY_INDIRECT      2
#define ACCESS_VARIABLE             3
#define ACCESS_VARIABLE_INDIRECT    4

class OPERAND : public PList
    {
    public:
        OPERAND();
        bool IsTrueCondition();
        long GetValue();
        long* GetValuePointer();

        long m_lValue;
        int m_iAccessMode;
        PString m_strValue;
    };

class VARIABLE : public OPERAND
    {
    public:
        VARIABLE( char* pszName );

        PString m_strName;
    };

class INSTRUCTION : public PNode
    {
    public:
        INSTRUCTION( long lCode );
        virtual ~INSTRUCTION();
        virtual bool Execute();

        long m_lCode, m_lLine;
        PList* m_lpOwner;
        
        OPERAND m_Source, m_Target, m_Condition;
    };

typedef INSTRUCTION* INSTRUCTION_POINTER;

#define OPCODE_GOTO                  1
#define OPCODE_GOSUB                 2
#define OPCODE_STOP                  3
#define OPCODE_RETURN                4
#define OPCODE_ADD                   5
#define OPCODE_SUB                   6
#define OPCODE_MUL                   7
#define OPCODE_DIV                   8
#define OPCODE_MOD                   9
#define OPCODE_IF_THEN               10
#define OPCODE_IF_THEN_ELSE          11
#define OPCODE_IF_THEN_UNLESS        12
#define OPCODE_IF_THEN_PROVIDED      13
#define OPCODE_COPY                  14
#define OPCODE_WRITE                 15
#define OPCODE_READ                  16
#define OPCODE_DECLARATION           17
#define OPCODE_WHILE_DO              18
#define OPCODE_WHILE_DO_UNLESS       19
#define OPCODE_WHILE_DO_PROVIDED     20
#define OPCODE_UNLESS_DO             21
#define OPCODE_REPEAT_UNTIL          22
#define OPCODE_REPEAT_UNLESS         23
#define OPCODE_DO_WHILE              24
#define OPCODE_DO_UNTIL              25
#define OPCODE_DO_UNLESS             26
#define OPCODE_UNTIL_DO              27
#define OPCODE_NAND                  28
#define OPCODE_MAX                   29

#define OPERAND_GT              1
#define OPERAND_LT              2
#define OPERAND_EQ              3
#define OPERAND_NE              4
#define OPERAND_LE              5
#define OPERAND_GE              6

class T 
    {
    public:
        T();
        bool ReadSourcecode( char* filename );
        void Run();
        void Run( PNode* pFirst );
        bool Parse( char* szBuffer );
        bool ParseSingleLine( char* szBuffer );
        bool CheckNameRestrictions( char* pszName );
        bool LINE_NUMBER();
        bool CONTROL_LOGIC();
        bool BRANCH_STATEMENT();
        bool IF_FAMILY();
        bool WHILE_FAMILY();
        bool IO_STATEMENTS();
        bool WRITE();
        bool READ();
        bool DECLARATION();
        bool IS_KEYWORD( char* pszKeyword );
        bool IS_INTEGER( long* pResult );
        bool IS_STRING( PString& refString );
        bool IS_NAME( PString& refString );
        bool IS_DECDIGIT( long* pResult );
        bool IMPLEMENT_IO_STATEMENT( char* pszKeyword, int nOpcode );
        bool NUMERIC_INSTRUCTION( char* pName, char* pKeyword, int nOpcode );
        bool BRANCH_INSTRUCTION( char* pKeyword, int nOpcode, bool bTarget );
        INSTRUCTION* GET_INSTRUCTION(OPERAND& x);
        bool STOP();
        bool GOTO();
        bool GOSUB();
        bool RETURN();
        bool MATH_STATEMENT();
        bool ADD();
        bool SUB();
        bool MOD();
        bool COPY();
        bool DIV();
        bool MUL();
        bool IS_NUMERIC(OPERAND& o);
        bool IS_BOOLEAN_EXPRESSION(OPERAND& o);
        INSTRUCTION* STATEMENT(PList* lpTarget);

        INSTRUCTION* m_pCurrentInstruction;
        char* m_pszCursor;
        long m_lCurrentLine;
        PList* m_lpTarget;
        PList m_Instructions;

        
        // name restrictions
        int m_nMinNameLength;
        int m_nMaxNameLength;
        char m_cValidNameRange[2];
        int m_cCharFrequency[3];    // char, count, modulo
        bool m_bValidOpcodes[OPCODE_MAX];
        void SetVariableInstructions( int* p );
        void DumpValidInstructions();
    };

#define PUSH() char* pszCursorCopy = m_pszCursor
#define POP() m_pszCursor = pszCursorCopy

#endif // TMMLPTEALPAITAFNFAL_H

