#include "precomp.h"
#include "time.h"
#include "ctype.h"
#include "OPP.h"
#include "TMMLPTEALPAITAFNFAL.h"


RNG rng;

static char* InstructionNames[] = 
{
    "INVALID",
    "GOTO", // OPCODE_GOTO                  
    "GOSUB", // OPCODE_GOSUB                 
    "STOP", // OPCODE_STOP                  
    "RETURN", // OPCODE_RETURN                
    "ADD", // OPCODE_ADD                   
    "SUB", // OPCODE_SUB                   
    "MUL", // OPCODE_MUL                   
    "DIV", // OPCODE_DIV                   
    "MOD", // OPCODE_MOD                   
    "IF-THEN", // OPCODE_IF_THEN               
    "IF-THEN-ELSE", // OPCODE_IF_THEN_ELSE          
    "IF-THEN-UNLESS", // OPCODE_IF_THEN_UNLESS        
    "IF-THEN-PROVIDED", // OPCODE_IF_THEN_PROVIDED      
    "COPY", // OPCODE_COPY                  
    "WRITE", // OPCODE_WRITE                 
    "READ", // OPCODE_READ                  
    "DECLARATION", // OPCODE_DECLARATION           
    "WHILE-DO", // OPCODE_WHILE_DO              
    "WHILE-DO-UNLESS", // OPCODE_WHILE_DO_UNLESS       
    "WHILE-DO-PROVIDED", // OPCODE_WHILE_DO_PROVIDED     
    "UNLESS-DO", // OPCODE_UNLESS_DO             
    "REPEAT-UNTIL", // OPCODE_REPEAT_UNTIL          
    "REPEAT-UNLESS", // OPCODE_REPEAT_UNLESS         
    "DO-WHILE", // OPCODE_DO_WHILE              
    "DO-UNTIL", // OPCODE_DO_UNTIL              
    "DO-UNLESS", // OPCODE_DO_UNLESS             
    "UNTIL-DO", // OPCODE_UNTIL_DO              
    "NAND", // OPCODE_NAND                  
};


RNG::RNG()
{
    time_t ltime;
    time(&ltime);
    tm* today = localtime(&ltime);
    idum = today->tm_yday + (today->tm_year*365);
    idum = idum | (idum << today->tm_wday);
    idum *= today->tm_yday;

    idum2 = idum;
    for( j=32+7;j>=0;j--)
    {
        k = idum/53668;
        idum=40014*(idum-k*53668)-idum*12211;
        if( idum < 0) 
            idum += 2147483563;
        if( j < 32)
            iv[j] = idum;
    }    
    iy=iv[0];
}   

#define NDIV (1+2147483562/32)

long RNG::GetRandomNumber()
{
    k=idum/53668;
    idum=40014*(idum2-k*53668)-k*12211;
    if( idum < 0 )
        idum += 2147483563;
    k=idum2/52774;
    idum2 = 40692*(idum2-k*52774)-k*3791;
    if( idum2 < 0 )
        idum2 += 2147483399;
    j=iy/NDIV;
    if( j < 0 ) j = -j;
    iy = iv[j]-idum2;
    iv[j] = idum;

    return iy & 0xFFFFFF;
}

#define MAX_MEMORY 3927

static long Memory[MAX_MEMORY];
static PList Variables;

VARIABLE::VARIABLE( char* pszName )
    :   m_strName( pszName )
{
    if( !pszName )  *((int*)0)=0;
}

VARIABLE* GetOrCreateVariable(char* pszName)
{
    ENUMERATE(&Variables,VARIABLE,pO)
        if( stricmp((char*)pO->m_strName,pszName) == 0 )
            return pO;

    pO = new VARIABLE(pszName);
    Variables.AddTail(pO);
    return pO;
}

T t;

INSTRUCTION* T::GET_INSTRUCTION(OPERAND& x)
{
    long v = x.GetValue();

    ENUMERATE(m_lpTarget,INSTRUCTION,pI)
    {
        if( pI->m_lLine == v )
        {
            return pI;
        }
    }
    return false;
}

OPERAND::OPERAND()
{
    m_lValue = 0;
    m_iAccessMode = ACCESS_IMMEDIATE;
}

long OPERAND::GetValue()
{
    if( m_iAccessMode == ACCESS_IMMEDIATE )
        return m_lValue;

    if( m_iAccessMode == ACCESS_MEMORY )
        return Memory[m_lValue];

    if( m_iAccessMode == ACCESS_VARIABLE )
    {
        VARIABLE* pV = GetOrCreateVariable(m_strValue);
        return pV ? pV->GetValue() : 0;
    }
    
    if( m_iAccessMode == ACCESS_VARIABLE_INDIRECT )
    {
        VARIABLE* pV = GetOrCreateVariable(m_strValue);
        return pV ? Memory[pV->GetValue()] : 0;
    }

    return Memory[Memory[m_lValue]];
}

long* OPERAND::GetValuePointer()
{
    if( m_iAccessMode == ACCESS_IMMEDIATE )
        return NULL;

    if( m_iAccessMode == ACCESS_MEMORY )
        return &(Memory[m_lValue]);

    if( m_iAccessMode == ACCESS_VARIABLE )
    {
        VARIABLE* pV = GetOrCreateVariable(m_strValue);
        return pV ? pV->GetValuePointer() : 0;
    }
    if( m_iAccessMode == ACCESS_VARIABLE_INDIRECT )
    {
        VARIABLE* pV = GetOrCreateVariable(m_strValue);
        return pV ? &(Memory[pV->GetValue()]) : 0;
    }

    return &(Memory[Memory[m_lValue]]);
}

bool OPERAND::IsTrueCondition()
{
    if( m_lCount != 2 )
        return (GetValue() != 0);
    
    long a = ((OPERAND*)m_pHead)->GetValue();
    long b = ((OPERAND*)m_pTail)->GetValue();

    switch(m_lValue)
    {
    case OPERAND_GT:
        return a > b;
    case OPERAND_LT:
        return a < b;
    case OPERAND_EQ:
        return a == b;
    case OPERAND_NE:
        return a != b;
    case OPERAND_LE:
        return a <= b;
    case OPERAND_GE:
        return a >= b;
    }
    return false;
}

bool INSTRUCTION::Execute()
{
    long Source, *pTarget;

    switch( m_lCode )
    {
    case OPCODE_DECLARATION:
        {
            VARIABLE* pO = GetOrCreateVariable(m_Target.m_strValue);
            if( pO )
            {
                pO->m_lValue = m_Source.m_lValue;
                pO->m_iAccessMode = m_Source.m_iAccessMode;
                pO->m_strValue = m_Source.m_strValue;
            }
            return true;
        }
    case OPCODE_WRITE:
        if( m_Condition.m_lValue == 1 )
            putchar((char)m_Source.GetValue());
        else
            printf( "%ld ", m_Source.GetValue());
        return true;

    case OPCODE_READ:
        pTarget = m_Source.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, READ INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        
        if( m_Condition.m_lValue == 1 )
            *pTarget = getchar();
        else
            scanf("%ld", pTarget ); 
        return true;

    case OPCODE_IF_THEN_ELSE:
        if( m_Source.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        else
            ((INSTRUCTION*)(m_Condition.m_pHead))->Execute();
        return true;
    case OPCODE_IF_THEN:
        if( m_Source.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;
    case OPCODE_WHILE_DO:
        while( m_Source.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;
    case OPCODE_WHILE_DO_UNLESS:
        while( m_Source.IsTrueCondition() && !m_Condition.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;

    case OPCODE_WHILE_DO_PROVIDED:
        while( m_Source.IsTrueCondition() && m_Condition.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;

    case OPCODE_UNLESS_DO:
    case OPCODE_UNTIL_DO:
        while( !m_Source.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;

    case OPCODE_DO_UNTIL:
    case OPCODE_REPEAT_UNTIL:  
        do
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        while( !m_Source.IsTrueCondition() );
        return true;

    case OPCODE_REPEAT_UNLESS:
    case OPCODE_DO_UNLESS:
        if( !m_Source.IsTrueCondition() )
        {
            do
                ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
            while( !m_Source.IsTrueCondition() );
        }
        return true;

    case OPCODE_DO_WHILE:
        do
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        while( m_Source.IsTrueCondition() );
        return true;

    case OPCODE_IF_THEN_UNLESS:
        if( m_Source.IsTrueCondition() && !m_Condition.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;
    case OPCODE_IF_THEN_PROVIDED:
        if( m_Source.IsTrueCondition() && m_Condition.IsTrueCondition() )
            ((INSTRUCTION*)(m_Target.m_pHead))->Execute();
        return true;
    case OPCODE_GOSUB:
        {
            INSTRUCTION* pI = t.GET_INSTRUCTION(m_Target);
            t.Run(pI);
        }
        return true;
    case OPCODE_GOTO:
        t.m_pCurrentInstruction = t.GET_INSTRUCTION(m_Target);
        return true;
    case OPCODE_RETURN:
        return false;
    case OPCODE_STOP:
        return false;
    case OPCODE_ADD:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        *pTarget += Source;
        return true;

    case OPCODE_SUB:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        *pTarget -= Source;
        return true;

    case OPCODE_MUL:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        *pTarget *= Source;
        return true;

    case OPCODE_NAND:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, NAND INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        *pTarget = ~(*pTarget) && ~(Source);
        return true;

    case OPCODE_DIV:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        if( !Source )
        {
            printf ("HEY, DIVISION BY ZERO IS A VERY BAD IDEA" );
            return false;
        }
        *pTarget /= Source;
        return true;
    case OPCODE_COPY:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        if( !IsEmptyString((char*)m_Source.m_strValue) && (m_Source.m_iAccessMode != ACCESS_VARIABLE) && (m_Source.m_iAccessMode != ACCESS_VARIABLE_INDIRECT) )
        {
            for( char* p = m_Source.m_strValue; *p; p++ )
            {
                *pTarget++ = *p;
            }
        }
        else
        {
            *pTarget = m_Source.GetValue();
        }
        return true;

    case OPCODE_MOD:
        pTarget = m_Target.GetValuePointer();
        if( !pTarget )
        {
            printf( "ERROR, ARITHMETIC INSTRUCTION MUST HAVE MEMORY TARGET, STUPID!" );
            return false;
        }
        Source = m_Source.GetValue();
        if( !Source )
        {
            printf ("HEY, MODULO ZERO IS A VERY BAD IDEA" );
            return false;
        }
        *pTarget %= Source;
        return true;
    }
    return false;
}

INSTRUCTION::INSTRUCTION( long lCode )
    :   m_lCode( lCode )
{
    t.m_lpTarget->AddTail(this);
    m_lpOwner = t.m_lpTarget;
    m_lLine = t.m_lCurrentLine;
}

INSTRUCTION::~INSTRUCTION()
{
    m_lpOwner->Remove(this);
}

bool T::IS_KEYWORD( char* pszKeyword )
{
    PUSH();

    SkipWhitespaces();

    int nLen = strlen(pszKeyword);
    if( strncmp(m_pszCursor,pszKeyword,nLen) == 0 )
    {
        if( isKeywordSeparator(m_pszCursor[nLen]) )
        {
            m_pszCursor += nLen;
            SkipWhitespaces();
            return true;
        }
    }
    POP();
    return false;
}

bool T::IS_DECDIGIT( long* pResult )
{
    if( (*m_pszCursor >= '0') && (*m_pszCursor <= '9') )
    {
        *pResult = *m_pszCursor - '0';
        m_pszCursor++;
        return true;
    }
    return false;
}

bool T::IS_INTEGER( long* pResult )
{
    PUSH();
    *pResult = 0;
    long digit = 0;
    SkipWhitespaces();
    if( IS_DECDIGIT(&digit) )
    {
        do
        {
            *pResult = *pResult *10;
            *pResult += digit;
        }
        while( IS_DECDIGIT(&digit) );
        SkipWhitespaces();
        return true;
    }

    POP();
    return false;
}

bool T::MATH_STATEMENT()
{
    return  NUMERIC_INSTRUCTION("ADD","TO",OPCODE_ADD) ||
            NUMERIC_INSTRUCTION("SUB","FROM",OPCODE_SUB) ||
            NUMERIC_INSTRUCTION("MOD","BY",OPCODE_MOD) ||
            NUMERIC_INSTRUCTION("DIV","BY",OPCODE_DIV) ||
            NUMERIC_INSTRUCTION("MUL","WITH",OPCODE_MUL) ||
            NUMERIC_INSTRUCTION("NAND","WITH",OPCODE_NAND) ||
            NUMERIC_INSTRUCTION("COPY","TO",OPCODE_COPY);
}

bool T::NUMERIC_INSTRUCTION( char* pName, char* pKeyword, int nOpcode )
{
    PUSH();
    if( IS_KEYWORD(pName) )
    {
        INSTRUCTION* pAI = new INSTRUCTION(nOpcode);
        if( IS_NUMERIC(pAI->m_Source) )
            if( IS_KEYWORD(pKeyword) )
                if( IS_NUMERIC(pAI->m_Target) )
                    return true;
        delete pAI;
    }
    POP();
    return false;
}

bool T::IO_STATEMENTS()
{
    return IMPLEMENT_IO_STATEMENT("READ",OPCODE_READ) || IMPLEMENT_IO_STATEMENT("WRITE",OPCODE_WRITE);
}

bool T::IMPLEMENT_IO_STATEMENT( char* pszKeyword, int nOpcode )
{
    PUSH();
    if( IS_KEYWORD(pszKeyword) )
    {
        INSTRUCTION* pI = new INSTRUCTION(nOpcode);
        if( IS_KEYWORD("CHAR") )
            pI->m_Condition.m_lValue = 1;
        else if( IS_KEYWORD("INTEGER") )
            pI->m_Condition.m_lValue = 2;
        else
            pI->m_Condition.m_lValue = 0;
        if( pI->m_Condition.m_lValue )
            if( IS_NUMERIC(pI->m_Source) )
                return true;
        delete pI;
    }
    POP();
    return false;
}

bool T::IS_BOOLEAN_EXPRESSION(OPERAND& o)
{
    OPERAND* p1 = new OPERAND();
    OPERAND* p2 = new OPERAND();
    if( IS_NUMERIC(*p1) )
    {
        if( IS_KEYWORD(">=") ) o.m_lValue = OPERAND_GE;
        else if( IS_KEYWORD("<=") ) o.m_lValue = OPERAND_LE;
        else if( IS_KEYWORD("=") ) o.m_lValue = OPERAND_EQ;
        else if( IS_KEYWORD("<>") ) o.m_lValue = OPERAND_NE;
        else if( IS_KEYWORD("<") ) o.m_lValue = OPERAND_LT;
        else if( IS_KEYWORD(">") ) o.m_lValue = OPERAND_GT;
        else 
        {
            o.m_lValue = p1->m_lValue;
            o.m_iAccessMode = p1->m_iAccessMode;
            o.m_strValue = p1->m_strValue;
            delete p1;
            delete p2;
            return true;
        }
        if( IS_NUMERIC(*p2) )
        {
            o.AddTail(p1);
            o.AddTail(p2);
            return true;
        }
    }

    delete p1;
    delete p2;
    return false;
}

bool T::IS_NUMERIC(OPERAND& o)
{
    PUSH();
    SkipWhitespaces();

    if( IS_INTEGER(&o.m_lValue) )
        return true;

    if( IS_STRING(o.m_strValue) )
        return true;

    if( IS_KEYWORD("CELL") )
        if( IS_INTEGER(&o.m_lValue) )
        {
            if( IS_KEYWORD("INDIRECT") )
                o.m_iAccessMode = ACCESS_MEMORY_INDIRECT;
            else
                o.m_iAccessMode = ACCESS_MEMORY;
            return true;
        }

    if( IS_NAME(o.m_strValue) )
    {
        if( IS_KEYWORD("INDIRECT") )
            o.m_iAccessMode = ACCESS_VARIABLE_INDIRECT;
        else
            o.m_iAccessMode = ACCESS_VARIABLE;
        return true;
    }

    POP();
    return false;
}

bool T::CheckNameRestrictions( char* pszName )
{
    if( (m_nMinNameLength != -1) && ((int)strlen(pszName) < m_nMinNameLength) )
    {
        printf( "ERROR, \"%s\" is not a valid name; identifiers must be at least %d characters long.", pszName, m_nMinNameLength );
        return false;
    }

    if( (m_nMaxNameLength != -1) && ((int)strlen(pszName) > m_nMaxNameLength) )
    {
        printf( "ERROR, \"%s\" is not a valid name; identifiers must be at less than %d characters long.", pszName, m_nMaxNameLength );
        return false;
    }

    if( m_cValidNameRange[0] && m_cValidNameRange[1] )
    {
        for( char* p = pszName; *p; p++ )
        {
            if( (*p < m_cValidNameRange[0]) || (*p > m_cValidNameRange[1]) )
            {
                printf( "ERROR, \"%s\" is not a valid name; characters must be in ASCII range %d-%d.", pszName, (int) m_cValidNameRange[0], (int) m_cValidNameRange[1] );
                return false;
            }
        }
    }
    if( m_cCharFrequency[0] && m_cCharFrequency[1] && m_cCharFrequency[2] )
    {
        int nCount = 0; 
        for( char* p = pszName; *p; p++ )
            if( *p == (char) m_cCharFrequency[0] )
                nCount++;

        nCount %= m_cCharFrequency[2];
        if( nCount != m_cCharFrequency[1] )
        {
            printf( "ERROR, \"%s\" is not a valid name; names must contain '%c' %d times modulo %d.",
                pszName, (char) m_cCharFrequency[0], m_cCharFrequency[1], m_cCharFrequency[2] );
            return false;
        }
    }
    return true;
}

bool T::IS_NAME( PString& refString )
{
    PUSH();
    if( isValidChar(*m_pszCursor) )
    {
        char* pszStart = m_pszCursor;
        while( isValidChar(*m_pszCursor) )
            m_pszCursor++;

        char c = *m_pszCursor;
        *m_pszCursor = 0;

        if( !CheckNameRestrictions(pszStart) )
            return false;

        refString = pszStart;
        *m_pszCursor = c;
        m_pszCursor++;
        return true;
    }
    POP();
    return false;
}

bool T::IS_STRING( PString& refString )
{
    PUSH();
    if( *m_pszCursor == '"' )
    {
        m_pszCursor++;
        char* pszStart = m_pszCursor;
        while( *m_pszCursor && (*m_pszCursor != '"') )
        {
            if( *m_pszCursor == '\\' )
            {
                if( (m_pszCursor[1] == '\\') || (m_pszCursor[1] == '"') )
                    m_pszCursor+=2;
            }
            else
            {
                m_pszCursor++;
            }
        }
        if( *m_pszCursor == '"' )
        {
            *m_pszCursor = 0;
            refString = pszStart;
            *m_pszCursor = '"';
            m_pszCursor++;
            return true;
        }
    }
    POP();
    return false;
}

bool T::BRANCH_INSTRUCTION( char* pKeyword, int nOpcode, bool bTarget )
{
    if( IS_KEYWORD(pKeyword) )
    {
        INSTRUCTION* pBI = new INSTRUCTION(nOpcode);
        if(!bTarget || IS_NUMERIC(pBI->m_Target))
            return true;
        delete pBI;
    }
    return false;
}

bool T::BRANCH_STATEMENT()
{
    return  BRANCH_INSTRUCTION("STOP",OPCODE_STOP,false) ||
            BRANCH_INSTRUCTION("RETURN",OPCODE_RETURN,false) ||
            BRANCH_INSTRUCTION("GOTO",OPCODE_GOTO,true) ||
            BRANCH_INSTRUCTION("GOSUB",OPCODE_GOSUB,true);
}

bool T::IF_FAMILY()
{
    PUSH();
    if( IS_KEYWORD("IF") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_IF_THEN);
        if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
            if( IS_KEYWORD("THEN") )
                if( STATEMENT(&(pI->m_Target)))
                    if( IS_KEYWORD("ELSE") )
                    {
                        pI->m_lCode = OPCODE_IF_THEN_ELSE;
                        if( STATEMENT(&(pI->m_Condition)) )
                            return true;
                    }
                    else if( IS_KEYWORD("UNLESS") )
                    {
                        pI->m_lCode = OPCODE_IF_THEN_UNLESS;
                        if( IS_BOOLEAN_EXPRESSION(pI->m_Condition) )
                            return true;
                    }
                    else if( IS_KEYWORD("PROVIDED") )
                    {
                        pI->m_lCode = OPCODE_IF_THEN_PROVIDED;
                        if( IS_BOOLEAN_EXPRESSION(pI->m_Condition) )
                            return true;
                    }
                    else return true;
        delete pI;
    }
    POP();
    return false;
}

bool T::WHILE_FAMILY()
{
    PUSH();
    if( IS_KEYWORD("WHILE") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_WHILE_DO);
        if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
            if( IS_KEYWORD("DO") )
                if( STATEMENT(&(pI->m_Target)))
                    if( IS_KEYWORD("UNLESS") )
                    {
                        pI->m_lCode = OPCODE_WHILE_DO_UNLESS;
                        if( IS_BOOLEAN_EXPRESSION(pI->m_Condition) )
                            return true;
                    }
                    else if( IS_KEYWORD("PROVIDED") )
                    {
                        pI->m_lCode = OPCODE_WHILE_DO_PROVIDED;
                        if( IS_BOOLEAN_EXPRESSION(pI->m_Condition) )
                            return true;
                    }
                    else return true;
        delete pI;
    }
    else if( IS_KEYWORD("DO") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_DO_WHILE);
        if( STATEMENT(&(pI->m_Target)) )
            if( IS_KEYWORD("WHILE") )
            {
                if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
                    return true;
            }
            else if( IS_KEYWORD("UNTIL") )
            {
                pI->m_lCode = OPCODE_DO_UNTIL;
                if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
                    return true;
            }
            else if( IS_KEYWORD("UNLESS") )
            {
                pI->m_lCode = OPCODE_DO_UNLESS;
                if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
                    return true;
            }

        delete pI;
    }
    else if( IS_KEYWORD("REPEAT") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_REPEAT_UNTIL);
        if( STATEMENT(&(pI->m_Target)) )
            if( IS_KEYWORD("UNTIL") )
            {
                pI->m_lCode = OPCODE_DO_UNTIL;
                if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
                    return true;
            }
            else if( IS_KEYWORD("UNLESS") )
            {
                pI->m_lCode = OPCODE_REPEAT_UNLESS;
                if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
                    return true;
            }

        delete pI;
    }
    else if( IS_KEYWORD("UNTIL") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_UNTIL_DO);
        if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
            if( IS_KEYWORD("DO") )
                if( STATEMENT(&(pI->m_Target)))
                    return true;
        delete pI;
    }
    else if( IS_KEYWORD("UNLESS") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_UNLESS_DO);
        if( IS_BOOLEAN_EXPRESSION(pI->m_Source) )
            if( IS_KEYWORD("DO") )
                if( STATEMENT(&(pI->m_Target)))
                    return true;
        delete pI;
    }
    POP();
    return false;
}

bool T::CONTROL_LOGIC()
{
    return BRANCH_STATEMENT() || IF_FAMILY() || WHILE_FAMILY();
}

bool T::DECLARATION()
{
    PUSH();
    if( IS_KEYWORD("DECLARE") )
    {
        INSTRUCTION* pI = new INSTRUCTION(OPCODE_DECLARATION);
        if( IS_NUMERIC(pI->m_Source) )
            if( IS_KEYWORD("AS") )
                if( IS_NAME(pI->m_Target.m_strValue) )
                    return true;
        delete pI;
    }
    POP();
    return false;
}

bool T::LINE_NUMBER()
{
    PUSH();
    long l;
    if( IS_KEYWORD("LINE") )
        if( IS_INTEGER(&l) )
            if( IS_KEYWORD(":") )
            {
                m_lCurrentLine = l;
                return true;
            }
    POP();
    return false;
}

INSTRUCTION* T::STATEMENT(PList* lpTarget)
{
    m_lpTarget = lpTarget;
    PNode* pLast = m_lpTarget->m_pTail;
    
    LINE_NUMBER();
    if( CONTROL_LOGIC() || MATH_STATEMENT() || IO_STATEMENTS() || DECLARATION() )
    {
        if( pLast != m_lpTarget->m_pTail )
        {
            INSTRUCTION* pI = (INSTRUCTION*)m_lpTarget->m_pTail;
            if( m_bValidOpcodes[pI->m_lCode] )
                return pI;
            printf( "ERROR, instructions of type %s are not allowed today.", InstructionNames[pI->m_lCode]);

            return NULL;
        }
    }
    return NULL;
}

bool T::ParseSingleLine( char* szBuffer )
{
    m_pszCursor = szBuffer;
    SkipWhitespaces();
    
    char* pComment = strstr( szBuffer, "//");
    if( pComment )
        *pComment = 0;

    if( !*m_pszCursor )
        return true;

    return (STATEMENT(&m_Instructions) != NULL);
}

bool T::ReadSourcecode( char* filename )
{
    PString strFilename(0,"%s.opp",filename);
    char* pExt = strrchr((char*)strFilename,'.');
    if(pExt)
    {
        *pExt = ',';

        OPP opp;
        if( !opp.ProcessFile(filename,(char*)strFilename) )
            return false;

        filename = strFilename;
    }

    FILE* fp = fopen(filename,"rt");
    if(fp)
    {
        char* buffer = (char*) calloc(1,291793), *write_pos,
            *read_pos, *line = (char*) calloc(1,1024);

        write_pos = buffer;

        while( !feof(fp) && fgets(line,1024,fp) )
        {
            read_pos = line;
            strcpy(write_pos,read_pos);
            write_pos += strlen(write_pos);
        }
        *write_pos = 0;
        fclose(fp);

        bool success = Parse(buffer);
        free(buffer);
        return success;
    }
    return false;
}

bool T::Parse( char* szBuffer )
{
    char* szStartOfLine = szBuffer;
    m_lCurrentLine = 0;
    while( *szBuffer )
    {
        if( *szBuffer == '\r' )
            *szBuffer = 0;
        else if( *szBuffer == '\n' )
        {
            *szBuffer = 0;
            m_lCurrentLine++;
            if( !ParseSingleLine(szStartOfLine) )
            {
                printf( "ERROR in line %ld : user lacks knowledge, probably\n>> \"%s\"", m_lCurrentLine, szStartOfLine );
                return false;
            }
            szStartOfLine = szBuffer+1;
        }
        szBuffer++;
    }
    m_lCurrentLine++;
    if( !ParseSingleLine(szStartOfLine) )
    {
        printf( "ERROR in line %ld : user lacks knowledge, probably\n>> \"%s\"", m_lCurrentLine, szStartOfLine );
        return false;
    }
    return true;
}

void T::Run( PNode* pFirst )
{
    INSTRUCTION* pI = m_pCurrentInstruction;
    m_pCurrentInstruction = (INSTRUCTION*) pFirst;
    while( m_pCurrentInstruction )
    {
        if( !m_pCurrentInstruction->Execute() )
            break;

        if( !m_pCurrentInstruction )
            m_pCurrentInstruction = (INSTRUCTION*) pFirst;
        else
            m_pCurrentInstruction = (INSTRUCTION*) (m_pCurrentInstruction->m_pNext);
    }
    m_pCurrentInstruction = pI;
}

void T::Run()
{
    Run( m_Instructions.m_pHead );
}

void T::SetVariableInstructions( int* p )
{
    int nCount = 0, nIndex, nNumberOfInstructions;

    for( nIndex = 0; p[nIndex] != -1; nIndex++ )
    {
        m_bValidOpcodes[p[nIndex]] = false;
        nCount++;
    }

    if( nCount == 2 )
        nNumberOfInstructions = 1;
    else
        nNumberOfInstructions = 1 + (rng.GetRandomNumber() % nCount);

    for( nIndex = 0; nIndex < nNumberOfInstructions; nIndex++ )
        m_bValidOpcodes[p[rng.GetRandomNumber() % nCount]] = true;
}

void T::DumpValidInstructions()
{
    printf( "VALID TMMLPTEALPAITAFNFAL INSTRUCTIONS FOR TODAY:\n" );
    for( int i = 1; i < OPCODE_MAX; i++ )
    {
        if( m_bValidOpcodes[i] )
            printf( "- %s\n", InstructionNames[i] );
    }

    printf( "RESTRICTIONS ON IDENTIFIERS FOR TODAY:\n" );
    if( m_nMinNameLength != -1 )
        printf( "NAMES MUST BE AT LEAST %d CHARS LONG\n", m_nMinNameLength  );

    if( m_nMaxNameLength != -1 )
        printf( "NAMES MUST BE LESS THAN %d CHARS LONG\n", m_nMaxNameLength  );

    if( m_cValidNameRange[0] )
        printf( "IDENTIFIER CHARACTERS MUST BE IN ASCII RANGE %d .. %d ('%c' .. '%c')\n",
        (int) m_cValidNameRange[0], (int) m_cValidNameRange[1], 
        (char) m_cValidNameRange[0], (char) m_cValidNameRange[1] );
     
    if( m_cCharFrequency[0] )
        printf( "CHARACTER %d ('%c') MUST BE USED %d TIMES MODULO %d PER IDENTIFIER\n",
            m_cCharFrequency[0], (char)m_cCharFrequency[0],
            m_cCharFrequency[1], m_cCharFrequency[2] );
}

T::T()
{
    if( rng.GetRandomNumber() % 2 )
        m_nMinNameLength = -1;
    else
        m_nMinNameLength = rng.GetRandomNumber() % 40;

    if( rng.GetRandomNumber() % 2 )
        m_nMaxNameLength = -1;
    else
        m_nMaxNameLength = m_nMinNameLength + rng.GetRandomNumber() % 100;

    if( rng.GetRandomNumber() % 2 )
        m_cValidNameRange[0] = m_cValidNameRange[1] = 0;
    else
    {
        m_cValidNameRange[0] = (rng.GetRandomNumber() % 30)+' ';
        m_cValidNameRange[1] = m_cValidNameRange[0] + (rng.GetRandomNumber() % 30)+' ';
    }                       

    if( rng.GetRandomNumber() % 2 )
        m_cCharFrequency[0] = m_cCharFrequency[1] = m_cCharFrequency[2] = 0;
    else
    {
        m_cCharFrequency[0] = (rng.GetRandomNumber() % 60)+' ';
        m_cCharFrequency[2] = rng.GetRandomNumber() % 60+1;
        m_cCharFrequency[1] = rng.GetRandomNumber() % m_cCharFrequency[2];
        
    }

    memset(m_bValidOpcodes,true,sizeof(m_bValidOpcodes));

    static int CII1[] = {
        OPCODE_GOTO, 
        OPCODE_GOSUB, -1 };
    SetVariableInstructions(CII1);

    static int CII2[] = {
        OPCODE_DIV,
        OPCODE_MOD, -1 };
    SetVariableInstructions(CII2);

   static int CII3[] = {
        OPCODE_IF_THEN,
        OPCODE_IF_THEN_ELSE,
        OPCODE_IF_THEN_UNLESS,
        OPCODE_IF_THEN_PROVIDED,
        OPCODE_WHILE_DO,
        OPCODE_WHILE_DO_UNLESS,
        OPCODE_WHILE_DO_PROVIDED,
        OPCODE_UNLESS_DO,
        OPCODE_REPEAT_UNTIL,
        OPCODE_REPEAT_UNLESS,
        OPCODE_DO_WHILE,
        OPCODE_DO_UNTIL,
        OPCODE_DO_UNLESS,
        OPCODE_UNTIL_DO, -1 };
    SetVariableInstructions(CII3);
}

int main( int argc, char* argv[] )
{
    if( argc == 2 )
    {
        if( stricmp(argv[1],"HELP") == 0 )
        {
            t.DumpValidInstructions();
        }
        else
        {
            if( t.ReadSourcecode(argv[1]) )
                t.Run();
            else printf( "UNABLE TO READ FILE %s\n", argv[1] );
            t.m_Instructions.RemoveContents();
        }
        
    }
    else printf( "USAGE: TMMLPTEALPAITAFNFAL <Sourcecode>|HELP\n" );
	return 0;
} // main()



