import string, statements, re, genericparser

""" Easy-IO/Method.

SYNTAX:

SOURCE      = STATEMENT {STATEMENT}.
STATEMENT   = CODE ['.'|'\n'].
CODE        = FUNCTION ':' DESCRIPTION.
FUNCTION    = FUNCNAME '(' [ARG {',' ARG}] ')'.
FUNCNAME    = NAME | BUILTIN.
BUILTIN     = 'if','while','until'...
NAME        = CHAR { CHAR | DIGIT }.
CHAR        = 'a' .. 'z'.
DIGIT       = '0' .. '9'.
DESCRIPTION = CODEM {'.' CODEM}
CODEM       = ASSIGNMENT | FUNCCALL | PRINT | EXPRESSION.
FUNCCALL    = NAME '(' [EXPRESSION {',' EXPRESSION}] ')'.
PRINT       = '<<' EXPRESSION
ASSIGNMENT  = VARIABLE ('='|'+='|'-='|'*='|'/='|'%='|'|='|'&=') EXPRESSION.
EXPRESSION  = OP0 {('<'|'=='|'>'|'<='|'>='|'!=') OP0}.
OP0         = OP1 {('+'|'-')OP1}.
OP1         = OP2 {('*'|'/'|'%')OP2}.
OP2         = OP3 {('|')OP3}.
OP3         = OP4 {('&')OP4}.
OP4         = ['!'] ('(' EXPRESSION ')' | FUNCCALL | VARIABLE | LITERAL).
LITERAL     = DECINT.
DECINT      = DIGIT {DIGIT}.
VARIABLE    = NAME.
"""

class Easy(genericparser.GenericParser):
    """
    In typical Python-Style, this code is totally unintelligible.
    And remember, comments are for PERL $Wimps
    """    

    def __init__(self):
        genericparser.GenericParser.__init__(self)
        self.code = []
        self.split_funcs_regex = re.compile('\(|\)')
        self.builtin_funcs = [
                ["do",statements.StatementBlock,0],
                ["until",statements.UntilDoStatement,1],
                ["while",statements.WhileDoStatement,1],
                ["while-do-unless",statements.WhileDoUnlessStatement,2],
                ["if",statements.IfStatement,1],
                ["if-then-else",statements.IfThenElseStatement,2],
                ["if-then-unless",statements.IfThenUnlessStatement,2],
                ["if-then-given",statements.IfThenProvidedStatement,2]
            ]
        self.ops = []
        self.ops.append({})
        self.ops[-1]['<'] = statements.IsLessStatement
        self.ops[-1]['=='] = statements.IsEqualStatement
        self.ops[-1]['>'] = statements.IsGreaterStatement
        self.ops[-1]['<='] = statements.IsLessOrEqualStatement
        self.ops[-1]['!='] = statements.IsNotEqualStatement
        self.ops[-1]['>='] = statements.IsGreaterOrEqualStatement
        self.ops.append({})
        self.ops[-1]['+'] = statements.AddStatement
        self.ops[-1]['-'] = statements.SubStatement
        self.ops.append({})
        self.ops[-1]['*'] = statements.MulStatement
        self.ops[-1]['/'] = statements.DivStatement
        self.ops[-1]['%'] = statements.ModStatement
        self.ops.append({})
        self.ops[-1]['|'] = statements.OrStatement        
        self.ops.append({})
        self.ops[-1]['&'] = statements.AndStatement
        self.assignments = [
                ["=",statements.AssignmentStatement],
                ["+=",statements.AddSelfStatement],
                ["*=",statements.MulSelfStatement],
                ["-=",statements.SubSelfStatement],
                ["/=",statements.DivSelfStatement],
                ["%=",statements.ModSelfStatement],
                ["&=",statements.AndSelfStatement],
                ["|=",statements.OrSelfStatement]
            ]
    
    def parse(self,filename):
        lines = open(filename).readlines()
        for i in range(len(lines)):
            if not self.parse_single_line(lines[i]):
                print "SYNTAX ERROR IN LINE", i+1
                return None
        if not isinstance(self.code,statements.StatementBlock):
            self.code = statements.StatementBlock(self.code)
        return self.code

    def parse_single_line(self,line):
        ci = line.find('#')
        if ci >= 0:
            line = line[:ci]
        line=map(string.strip,string.split(line,':'))
        if line == ['']: return 1
        if len(line) != 2: return 0
        function = self.FUNCTION(line[0])
        if function is None:
            print "ERROR, invalid function declaration"
            return 0
        description = self.DESCRIPTION(line[1])
        if description is None:
            print "ERROR, invalid code declaration"
            return 0

        if isinstance(function,statements.FuncDefStatement):
            function.instruction = description
        elif self.required_args == 0:
            if isinstance(description,statements.StatementBlock):
                function.statements = description.statements
            else:
                function.statements = [description]
        elif self.required_args == 1:
            function.args = description.statements[0]
            description.statements = description.statements[1:]
            function.code = description
        elif self.required_args == 2:
            function.args = description.statements[0]
            function.more = description.statements[1]
            description.statements = description.statements[2:]
            function.code = description
        self.code.append(function)       
        return 1

    def DESCRIPTION(self,expression):
        self.str,self.pos = expression,0
        result = [self.CODEM()]
        while (result[-1] is not None) and self.skip('.'):
            result.append(self.CODEM())
        if result[-1] is not None:
            if len(result) == 1:
                return result[0]
            return statements.StatementBlock(result)
        return None

    def EXPRESSION(self):
        return self.op_expression(0)

    def op_expression(self,oplevel):
        if oplevel >= len(self.ops):
            return self.OP4()
        a,pos = self.op_expression(oplevel+1),self.pos
        if a is not None:
            while 1:
                c = self.str[self.pos:self.pos+2]
                if not self.ops[oplevel].has_key(c):
                    c = self.peek()              
                    if not self.ops[oplevel].has_key(c):
                        return a
                o = self.ops[oplevel][c]()
                self.pos += len(c)
                b = self.op_expression(oplevel+1)
                if b is None: break
                o.source,o.target = a,b
                a = o
        self.pos = pos
        return None


    def OP4(self):
        pos = self.pos
        negate = 0
        if self.skip('!'):
            negate = 1      
        result = self.oneof(self.BRACKET_EXPRESSION,self.FUNCCALL,self.VARIABLE,self.LITERAL)      
        if result is not None:
            if not negate:
                return result
            return statements.NegateStatement(result)
        self.pos = pos
        return None

    def BRACKET_EXPRESSION(self):
        pos = self.pos
        if self.skip('('):
            result = self.EXPRESSION()
            if self.skip(')'):
                return result       
        self.pos = pos
        return None

    def CODEM(self):
        return self.oneof(self.ASSIGNMENT,self.FUNCCALL,self.PRINT,self.EXPRESSION)

    def PRINT(self):
        pos = self.pos
        if self.skip('<<'):
            i = self.EXPRESSION()
            if i is not None:
                return statements.PrintStatement(i)
        self.pos = pos
        return None

    def FUNCCALL(self):
        pos = self.pos
        n = self.NAME()
        if n is not None:
            if self.skip('('):
                e = self.EXPRESSION()
                if e is not None:
                    result = [e]
                    while self.skip(','):
                        e = self.EXPRESSION()
                        if e is not None:
                            result.append(e)
                        else: break
                    if self.skip(')'):
                        return statements.FuncCallStatement(n,result)

        self.pos = pos
        return None

    def ASSIGNMENT(self):
        pos = self.pos
        v = self.VARIABLE()
        if v is not None:
            for i in self.assignments:
                if self.skip(i[0]):
                    e = self.EXPRESSION()
                    if e is not None:
                        return i[1](v,e)
                    break

        self.pos = pos        
        return None

    def FUNCTION(self,expression):
        expression = self.split_funcs_regex.split(expression)
        
        if len(expression) == 1:
            for i in range(len(self.builtin_funcs)):
                if expression[0] == self.builtin_funcs[i][0]:
                    self.required_args = self.builtin_funcs[i][2]
                    return self.builtin_funcs[i][1]()
           
        if (len(expression) == 3) and self.ISVALIDNAME(expression[0]):
            func = statements.FuncDefStatement(expression[0])
            func.arguments = expression[1].split(',')
            return func

        return func

    def ISVALIDNAME(self,name):
        if name[0] in string.letters:
            for i in range(1,len(name)):
                if (name[i] not in string.letters) and (not name[i] in string.digits):
                    return 0
            return 1
        return 0
