import string, statements, re, genericparser

"""

Syntax:

FILE        = SEQUENCE
SEQUENCE    = '(' OBJECT [{ (',' OBJECT) }] ')'.
OBJECT      = '(' OBJECTDEF ')'.
OBJECTDEF   = ASSIGNMENT | WHILE | IF | WHILE_UNLESS | UNTIL_DO | IF_ELSE | IF_UNLESS | IF_PROVIDED | FUNCDEF | FUNCCALL | PRINT | EXPRESSION.
IF          = 'IF ' OBJECT ' DO ' OBJECT.
WHILE       = 'WHILE' OBJECT ' DO ' OBJECT.
WHILE_UNLESS = 'WHILE ' OBJECT ' DO ' OBJECT ' UNLESS ' OBJECT.
UNTIL_DO    = 'UNTIL ' OBJECT ' DO ' OBJECT.
IF_ELSE     = 'IF ' OBJECT ' THEN ' OBJECT ' ELSE ' OBJECT.
IF_UNLESS   = 'IF ' OBJECT ' THEN ' OBJECT ' UNLESS ' OBJECT.
IF_PROVIDED = 'IF ' OBJECT ' THEN ' OBJECT ' PROVIDED ' OBJECT.
PRINT       = '<<' EXPRESSION.
ASSIGNMENT  = VARIABLE ('='|'+='|'-='|'*='|'/='|'%='|'|='|'&=') (('(' EXPRESSION ')') | NAME).
FUNCDEF     = NAME '(' [NAME {',' NAME}] ')=' SEQUENCE.
FUNCCALL    = NAME '(' [EXPRESSION {',' EXPRESSION}] ')'
EXPRESSION  = OP0 {('<'|'=='|'>'|'<='|'>='|'!=') OP0}. ')'.
OP0         = OP1 {('+'|'-')OP1}.
OP1         = OP2 {('*'|'/'|'%')OP2}.
OP2         = OP3 {('|')OP3}.
OP3         = OP4 {('&')OP4}.
OP4         = ['!'] ('(' EXPRESSION ')' | '(' FUNCCALL ')' | VARIABLE | LITERAL).
LITERAL     = DECINT.
NAME        = CHAR { CHAR | DIGIT }.
CHAR        = 'a' .. 'z'.
DIGIT       = '0' .. '9'.
DECINT      = DIGIT {DIGIT}.


"""

class Brackets(genericparser.GenericParser):

    def dumpargs(self,args):
        result = '('
        for arg in args:
            if type(arg)==type("s"):
                result += arg + ','
            else:
                result += self.dump(arg) + ','
        return result[:-1] + ')'              

    def FuncDefStatement(self,x):
        return "(%s%s=%s)" % (x.name, self.dumpargs(x.arguments),self.dump(x.instruction))

    def FuncCallStatement(self,x):
        return "(%s%s)" % (x.name,self.dumpargs(x.args))

    def VariableStatement(self,x):
        return x.name

    def StatementBlock(self,x):
        global tablevel
        result = "("
        if len(x.statements):
            for s in x.statements:
                result += self.dump(s) + "," # ",\n"
            return result[:-2]+')'
        
        return result

    def AssignmentStatement(self,x):
        return "(%s=%s)" % (self.dump(x.source),self.dump(x.target))

    def AddSelfStatement(self,x):
        return "(%s+=%s)" % (self.dump(x.source),self.dump(x.target))

    def AddStatement(self,x):
        return "(%s+%s)" % (self.dump(x.source),self.dump(x.target))

    def SubSelfStatement(self,x):
        return "(%s-=%s)" % (self.dump(x.source),self.dump(x.target))

    def SubStatement(self,x):    
        return "(%s-%s)" % (self.dump(x.source),self.dump(x.target))

    def MulSelfStatement(self,x):
        return "(%s*=%s)" % (self.dump(x.source),self.dump(x.target))

    def MulStatement(self,x):
        return "(%s*%s)" % (self.dump(x.source),self.dump(x.target))

    def DivSelfStatement(self,x):
        return "(%s/=%s)" % (self.dump(x.source),self.dump(x.target))

    def DivStatement(self,x):
        return "(%s/%s)" % (self.dump(x.source),self.dump(x.target))

    def ModSelfStatement(self,x):
        return "(%s%%=%s)" % (self.dump(x.source),self.dump(x.target))

    def ModStatement(self,x):
        return "(%s%%%s)" % (self.dump(x.source),self.dump(x.target))

    def OrSelfStatement(self,x):
        return "(%s||=%s)" % (self.dump(x.source),self.dump(x.target))

    def OrStatement(self,x):
        return "(%s||%s)" % (self.dump(x.source),self.dump(x.target))

    def AndSelfStatement(self,x):
        return "(%s&&=%s)" % (self.dump(x.source),self.dump(x.target))

    def AndStatement(self,x):
        return "(%s&&%s)" % (self.dump(x.source),self.dump(x.target))

    def BitOrSelfStatement(self,x):
        return "(%s|=%s)" % (self.dump(x.source),self.dump(x.target))

    def BitOrStatement(self,x):
        return "(%s|%s)" % (self.dump(x.source),self.dump(x.target))

    def BitAndSelfStatement(self,x):
        return "(%s&=%s)" % (self.dump(x.source),self.dump(x.target))

    def BitAndStatement(self,x):
        return "(%s&%s)" % (self.dump(x.source),self.dump(x.target))

    def LiteralStatement(self,x):
        if type(x.value) == type(""):
            return '"%s"' % x.value
        return str(x.value)

    def PrintStatement(self,x):
        return "(<<%s)" % self.dump(x.value)

    def PrintCharStatement(self,x):
        return "(-<<%s)" % self.dump(x.value)
    
    def NegateStatement(self,x):
        return "!%s" % self.dump(x.value)

    def IfStatement(self,x):
        return "(IF %s DO %s)" % (self.dump(x.args),self.dump(x.code))
        
    def WhileDoStatement(self,x):
        return "(WHILE %s DO %s)" % (self.dump(x.args),self.dump(x.code))

    def IfThenElseStatement(self,x):
        return "(IF %s THEN %s ELSE %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))

    def IfThenUnlessStatement(self,x):
        return "(IF %s THEN %s UNLESS %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))
                                            
    def IfThenProvidedStatement(self,x):
        return "(IF %s THEN %s PROVIDED %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))

    def WhileDoUnlessStatement(self,x):
        return "(WHILE %s DO %s UNLESS %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))

    def ForStatement(self,x):
        return "(FOR %s IN %s DO %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))

    def WhileDoElseStatement(self,x):
        return "(WHILE %s DO %s ELSE %s)" % (self.dump(x.args),self.dump(x.code),self.dump(x.more))

    def UntilDoStatement(self,x):
        return "(UNTIL %s DO %s)" % (self.dump(x.args),self.dump(x.code))

    def IsLessStatement(self,x):
        return "(%s<%s)" % (self.dump(x.source),self.dump(x.target))

    def IsEqualStatement(self,x):
        return "(%s==%s)" % (self.dump(x.source),self.dump(x.target))

    def IsGreaterStatement(self,x):
        return "(%s>%s)" % (self.dump(x.source),self.dump(x.target))

    def IsLessOrEqualStatement(self,x):
        return "(%s<=%s)" % (self.dump(x.source),self.dump(x.target))

    def IsNotEqualStatement(self,x):
        return "(%s!=%s)" % (self.dump(x.source),self.dump(x.target))

    def IsGreaterOrEqualStatement(self,x):
        return "(%s>=%s)" % (self.dump(x.source),self.dump(x.target))

    def dump(self,x):
        sx = str(x)
        if sx[:12] == "<statements.":
            sx = sx[12:]
            ix = sx.find(' ')
            if ix > 0:
                sx = sx[:ix]
                o = getattr(self,sx)
                if o:
                    return o(x)

        return str(x)

    def __init__(self):
        genericparser.GenericParser.__init__(self)
        self.ops = []
        self.ops.append({})
        self.ops[-1]['<'] = statements.IsLessStatement
        self.ops[-1]['=='] = statements.IsEqualStatement
        self.ops[-1]['>'] = statements.IsGreaterStatement
        self.ops[-1]['<='] = statements.IsLessOrEqualStatement
        self.ops[-1]['!='] = statements.IsNotEqualStatement
        self.ops[-1]['>='] = statements.IsGreaterOrEqualStatement
        self.ops.append({})
        self.ops[-1]['+'] = statements.AddStatement
        self.ops[-1]['-'] = statements.SubStatement
        self.ops.append({})
        self.ops[-1]['*'] = statements.MulStatement
        self.ops[-1]['/'] = statements.DivStatement
        self.ops[-1]['%'] = statements.ModStatement
        self.ops.append({})
        self.ops[-1]['|'] = statements.OrStatement        
        self.ops.append({})
        self.ops[-1]['&'] = statements.AndStatement
        self.assignments = [
                ["=",statements.AssignmentStatement],
                ["+=",statements.AddSelfStatement],
                ["*=",statements.MulSelfStatement],
                ["-=",statements.SubSelfStatement],
                ["/=",statements.DivSelfStatement],
                ["%=",statements.ModSelfStatement],
                ["&=",statements.AndSelfStatement],
                ["|=",statements.OrSelfStatement]
            ]
    

    def parse(self,filename):
        lines = open(filename).readlines()
        expression = ""
        for line in lines:
            while len(line) and (line[-1] in ('\n','\t',' ')):
                line = line[:-1]
            while len(line) and (line[0] in ('\n','\t',' ')):
                line = line[1:]
            expression += line

        return self.parseString(expression)

    def parseString(self,expression):
        self.str,self.pos = expression,0
        return self.SEQUENCE()

    def SEQUENCE(self):
        self.skipwhitespaces()
        result = []
        pos = self.pos
        self.DEBUG("SEQUENCE BEGIN")
        self.tabs+=1
        if self.skip('('):
            o = self.SEQUENCE()
            if o is not None:
                result.append(o)
                while self.skip(','):
                    
                    o = self.SEQUENCE()
                    if o is None:
                        break
                    result.append(o)
                if o is not None:
                    self.skip(')')
                    self.tabs-=1
                    self.DEBUG("SEQUENCE RETURNS " + str(result))
                    if len(result) == 1:
                        return result[0]
                    return statements.StatementBlock(result)
        else:
            o = self.OBJECT()
            if o is not None:
                return o
        self.tabs-=1
        self.DEBUG("SEQUENCE FAILED")
        self.pos = pos
        return None
    
    def OBJECT(self):
        self.DEBUG("OBJECTDEF BEGIN")
        self.tabs+=1
        
        o = self.oneof(self.ASSIGNMENT, self.WHILE, self.IF, self.WHILE_UNLESS, self.WHILE_ELSE, self.FOR, self.UNTIL_DO,
            self.IF_ELSE, self.IF_UNLESS, self.IF_PROVIDED, self.PRINT, self.PRINTCHAR, self.FUNCDEF, self.FUNCCALL,self.EXPRESSION)

        self.tabs-=1
        self.DEBUG("OBJECT RETURNS "+str(o))
        return o

    def HandleKeywordList(self,stmt,*keywords):      
        self.DEBUG(str(stmt)+" BEGIN")
        self.tabs+=1
        
        result = []        
        pos = self.pos
        for i in range(len(keywords)):            
            self.skipwhitespaces()
            self.DEBUG("TESTING KEYWORD "+keywords[i])
            if not self.skip(keywords[i]):
                self.pos = pos
                self.tabs-=1
                self.DEBUG(str(stmt)+" RETURNS None AT #1")
                return None
            
            result.append( self.SEQUENCE() )
            self.DEBUG("OBJECT WAS "+str(result[-1]))
            
            if result[-1] is None:
                self.pos = pos
                self.tabs-=1
                self.DEBUG(str(stmt)+" RETURNS None AT #2")
                return None
            
        o = stmt()
        o.args = result[0]
        o.code = result[1]
        if len(result) == 3:
            o.more = result[2]

        self.tabs-=1
        self.DEBUG(str(stmt)+" RETURNS "+str(o))
        return o
        
    def WHILE(self):
        return self.HandleKeywordList(statements.WhileDoStatement,'WHILE','DO')
    
    def IF(self):
        return self.HandleKeywordList(statements.IfStatement,'IF','DO')
    
    def WHILE_UNLESS(self):
        return self.HandleKeywordList(statements.WhileDoUnlessStatement,'WHILE','DO','UNLESS')

    def WHILE_ELSE(self):
        return self.HandleKeywordList(statements.WhileDoElseStatement,'WHILE','DO','ELSE')
    
    def FOR(self):
        return self.HandleKeywordList(statements.ForStatement,'FOR','IN','DO')

    def UNTIL_DO(self):
        return self.HandleKeywordList(statements.UntilDoStatement,'UNTIL','DO')
    
    def IF_ELSE(self):
        return self.HandleKeywordList(statements.IfThenElseStatement,'IF','THEN','ELSE')
    
    def IF_UNLESS(self):
        return self.HandleKeywordList(statements.IfThenUnlessStatement,'IF','THEN','UNLESS')
    
    def IF_PROVIDED(self):
        return self.HandleKeywordList(statements.IfThenProvidedStatement,'IF','THEN','PROVIDED')
    
    def FUNCCALL(self):
        self.DEBUG("FUNCCALL BEGIN")
        self.tabs+=1        
        pos,n = self.pos, self.NAME()
        if n is not None:
            if self.skip('('):
                a = [self.EXPRESSION()]
                while a[-1] is not None:
                    if self.skip(','):
                        a.append(self.EXPRESSION())
                    else: break
                if self.skip(')'):
                    result = statements.FuncCallStatement(n,a)
                    self.tabs-=1
                    self.DEBUG("FUNCCALL RETURNS "+str(result))
                    return result
        self.tabs-=1
        self.DEBUG("FUNCCALL FAILS")
        self.pos = pos
        return None

    def PRINTCHAR(self):
        self.DEBUG("PRINTCHAR BEGIN")
        self.tabs+=1        
        if self.skip('-<<'):
            o = self.EXPRESSION()
            if o is not None:
                result = statements.PrintCharStatement(o)
                self.tabs-=1
                self.DEBUG("PRINTCHAR RETURNS "+str(result))
                return result
                
        self.tabs-=1        
        self.DEBUG("PRINT FAILS")
        return None
    
    def PRINT(self):
        self.DEBUG("PRINT BEGIN")
        self.tabs+=1        
        if self.skip('<<'):
            o = self.EXPRESSION()
            if o is not None:
                result = statements.PrintStatement(o)
                self.tabs-=1
                self.DEBUG("PRINT RETURNS "+str(result))
                return result
                
        self.tabs-=1        
        self.DEBUG("PRINT FAILS")
        return None
    
    def EXPRESSION(self):
        self.DEBUG("EXPRESSION BEGIN")
        self.tabs+=1

        o = self.op_expression(0)

        self.tabs-=1        
        self.DEBUG("EXPRESSION RETURNS "+str(o))        
        return o

    def op_expression(self,oplevel):
        if oplevel >= len(self.ops):
            return self.OP4()
        a,pos = self.op_expression(oplevel+1),self.pos
        if a is not None:
            while 1:
                c = self.str[self.pos:self.pos+2]
                if not self.ops[oplevel].has_key(c):
                    c = self.peek()              
                    if not self.ops[oplevel].has_key(c):
                        return a
                o = self.ops[oplevel][c]()
                self.pos += len(c)
                b = self.op_expression(oplevel+1)
                if b is None: break
                o.source,o.target = a,b
                a = o
        self.pos = pos
        return None

    def BRACKET_EXPRESSION(self):
        self.DEBUG("BRACKET_EXPRESSION BEGIN")
        self.tabs+=1        
        pos = self.pos
        if self.skip('('):
            result = self.EXPRESSION()
            if self.skip(')'):
                self.tabs-=1
                self.DEBUG("BRACKET_EXPRESSION RETURNS "+str(result))                
                return result
        self.tabs-=1
        self.DEBUG("BRACKET_EXPRESSION FAILS")
        self.pos = pos
        return None

    def BRACKET_FUNCCALL(self):
        self.DEBUG("BRACKET_FUNCCALL BEGIN")
        self.tabs+=1        
        pos = self.pos
        if self.skip('('):
            result = self.FUNCCALL()
            if self.skip(')'):
                self.tabs-=1
                self.DEBUG("BRACKET_FUNCCALL RETURNS "+str(result))
                return result       
        self.tabs-=1
        self.DEBUG("BRACKET_FUNCCALL FAILS")
        self.pos = pos
        return None

    def OP4(self):
        self.DEBUG("OP4 BEGIN")
        self.tabs+=1

        pos = self.pos
        negate = 0
        if self.skip('!'):
            negate = 1
        result = self.oneof(self.VARIABLE,self.LITERAL,self.BRACKET_EXPRESSION,self.BRACKET_FUNCCALL)
        if result is not None:
            if not negate:
                self.tabs-=1
                self.DEBUG("OP4 RETURNS "+str(result))
                return result                
            result = statements.NegateStatement(result)
            self.tabs-=1
            self.DEBUG("OP4 RETURNS "+str(result))
            return result                

        self.tabs-=1
        self.DEBUG("OP4 FAILS")        
        self.pos = pos
        return None

    def FUNCDEF(self):
        self.DEBUG("FUNCDEF BEGIN")
        self.tabs+=1
        pos,n = self.pos, self.NAME()
        if n is not None:
            if self.skip('('):
                a = [self.NAME()]
                while a[-1] is not None:
                    if self.skip(','):
                        a.append(self.NAME())
                    else: break
                if self.skip(')='):
                    s = self.SEQUENCE()
                    if s is not None:
                        result = statements.FuncDefStatement(n)
                        result.instruction = s
                        result.arguments = a
                        self.tabs-=1
                        self.DEBUG("FUNCDEF RETURNS "+str(result))
                        return result
        self.tabs-=1        
        self.DEBUG("FUNCDEF FAILS")
        self.pos = pos
        return None

    def ASSIGNMENT(self):
        self.DEBUG("ASSIGNMENT BEGIN")
        self.tabs+=1
        pos = self.pos
        v = self.VARIABLE()
        if v is not None:
            for i in self.assignments:
                if self.skip(i[0]):
                    e = self.EXPRESSION()
                    if e is not None:
                        o = i[1](v,e)
                        self.tabs-=1
                        self.DEBUG("ASSIGNMENT RETURNS "+str(o))
                        return o                        
                    break

        self.tabs-=1
        self.DEBUG("ASSIGNMENT FAILS")
        self.pos = pos        
        return None

