#
# spamcheck.py                                                             
#                                                                      
# Copyright (C) 2001,2002 Gerson Kurz (not@p-nand-q.com)
# BSD-Licensed. This is free software, enjoy!

import socket, types, fnmatch, re, pop3trace

"""
This module defines helper functions for checking a single IP address (or DNS name) against a list of
IP addresses/DNS names including regular-expressions, wildcards and explicit ranges.
"""

def getDnsName(hostaddr):
    "Returns the DNS name for a given host address"
    try:
        return socket.gethostbyaddr(hostaddr)[0]
    except:
        return hostaddr

def getIpAddress(hostname):
    "Returns the IP address for a given DNS name"
    try:
        return socket.gethostbyname(hostname)
    except:
        return hostname

def getAddressAsLong(ipaddr):
    "Converts an IP address to its LONG value"
    tokens = ipaddr.split(".")
    if len(tokens) != 4:
        raise "ERROR, IP-Address '%s' invalid" % ipaddr
    return reduce(lambda x,y:x+y,map(lambda x,y:long(x)*y,tokens,[256**i for i in range(3,-1,-1)]))

def isIpAddress(ipaddr):
    "If the input is an IP address, return its LONG value; otherwise, return None"
    try:
        return reduce(lambda x,y:x+y,map(lambda x,y:long(x)*y,ipaddr.split("."),[256**i for i in range(3,-1,-1)]))
    except:
        return None

# this dict is used as a cached for compiled regular expressions
precompiled_patterns = {}

# internal function to recursively enumerate all statements in a rule list
def _AddressIn( hostaddr_string, hostaddr_number, hostname, range ):
   
    if isinstance(range,types.ListType):

        # the "range" argument is a list of single address specifications
        for item in range:
            if _AddressIn( hostaddr_string, hostaddr_number, hostname, item ):
                return 1

        return 0        
        
    if isinstance(range,types.TupleType):

        # the "range" argument is an address range. It is assumed that ranges
        # are always specified explicitly by ip numbers
        assert(len(range)==2)

        r = map( getAddressAsLong, range )
        if hostaddr_number >= r[0] and hostaddr_number <= r[1]:
            return 1

        return 0

    assert(isinstance(range,types.StringType))

    if range[0] == '$':
        global precompiled_patterns

        # the "range" argument is a regular expression
        try:
            pattern = precompiled_patterns[range]
        except:
            pattern = re.compile(range[1:])
            precompiled_patterns[range] = pattern

        if pattern.match(hostaddr_string) or pattern.match(hostname):
            return 1

    elif range.find("*") or range.find("?") or range.find("["):

        # use simple wildcard matching
        if fnmatch.fnmatch(hostaddr_string, range) or fnmatch.fnmatch(hostname,range):
            return 1

    elif range == hostaddr_string or range == hostname:
        return 1

    return 0            
                
def addressInRange( hostaddr, range ):
    """
    This function checks if the <hostaddr> is part of the <range> specification, which
    can be either a single address, a list of addresses or address ranges (including wildcards).
    hostaddr can be either an IP address (form a.b.c.d), or a dns name.
    """
    hostaddr_string = hostaddr
    temp = isIpAddress(hostaddr)
    if temp is not None:
        hostaddr_number = temp
        hostname = getDnsName(hostaddr)
    else:
        index = hostaddr.rfind("@")
        if index >= 0:
            hostaddr = hostaddr[index+1:]
        hostname = hostaddr
        try:
            hostaddr_string = getIpAddress(hostname)
            hostaddr_number = getAddressAsLong(hostaddr_string)
        except:
            pop3trace.trace( "ERROR, unable to resolve '%s'" % hostname )
            return 0

    return _AddressIn( hostaddr_string, hostaddr_number, hostname, range )

def decodeAddressRangeString(data):
    """
    This decodes an address-range string. The string is a ; separated list of items. Each item can be

        - a pattern
        - a regular expression, if prefixed by '$'
        - a tuple
    """
    result = []
    try:
        for item in data.split(";"):
            if not len(item):
                continue

            if item[0] == '(' and item[-1] == ')':
                item = item.split("-")
                if len(item) == 2:
                    result.append( (item[0][1:], item[1][:-1]) )
                else:
                    pop3trace.trace("ERROR, unkown address spec '%s'" % item)
            else:
                result.append(item)
    except:
        pass
    return result

# make sure the latest spam rules are read
# HACK to fool py2exe:
import_this = list('import spamtokens')
try:
    exec("".join(import_this))        
except:
    pass

def checkRelayBlacklist(ip):

    "This function checks a (wellformed) IP address against spam blacklists."
    if addressInRange(ip, spamtokens.BLACKLIST_IGNORE):
        pop3trace.trace("IP address '%s' will not be checked because it is in the blacklist-ignore table" % ip)
        return None
    
    ip = ip.split(".")
    ip.reverse()
    ip = ".".join(ip) + "."
    for domain in spamtokens.BLACKLIST_DOMAINS:
        host = ip + domain 
        try:
            pop3trace.trace("Checking for DNS entry '%s'" % host)
            addr = socket.gethostbyname(host)
        except socket.error:
            continue
        if addr:
            return domain
    return None

